/** @file   Color.h
 * @brief   Declaration and implementation of Color class
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:37 $
 * @author  Tomi Lamminsaari
 */

 
#ifndef H_COLOR_H
#define H_COLOR_H

#include <allegro.h>
#include "eng2d_dll.h"
#include "datatypes.h"

namespace eng2d {

/** @class  Color
 * @brief   Represents a rgb-color triplet.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 *
 * 
 */
class DLLIMPORT Color
{
public:
  ///
  /// Static members and methods
  /// ==========================
    
  
  ///
  /// Members
  /// =======

  /** The red component. */
  CVal m_r;
  /** The green component. */
  CVal m_g;
  /** The blue component. */
  CVal m_b;
  
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs black color.
   */
  inline Color() :
    m_r( 0 ),
    m_g( 0 ),
    m_b( 0 )
  {
  }
  
  
  /** Constructs new color.
   * @param     allegroC          Should have value you've got from Allegro's
   *                              makecol(...)-function.
   */
  inline Color(int allegroC)
  {
    m_r = getr( allegroC );
    m_g = getg( allegroC );
    m_b = getb( allegroC );
  }
  
  
  /** Constructs new color based on given rgb-triplet.
   * @param     r                 The red value
   * @param     g                 The green value.
   * @param     b                 The blue value
   */
  inline Color( CVal r, CVal g, CVal b) :
    m_r( r ),
    m_g( g ),
    m_b( b )
  {
  }
  
  /** A copy constructor.
   * @param     rObject           Another color
   */
  inline Color(const Color& rObject) :
    m_r( rObject.m_r ),
    m_g( rObject.m_g ),
    m_b( rObject.m_b )
  {
  }
  
  /** Destructor.
   */
  inline ~Color()
  {
  }
  
  
  /** Assignment operator
   */
  inline Color& operator = (const Color& rO)
  {
    m_r = rO.m_r;
    m_g = rO.m_g;
    m_b = rO.m_b;
    return *this;
  }
  
  /** Operator +. Adds the color-components to each other.
   */
  inline Color operator + (const Color& rO) const
  {
    return Color( (m_r + rO.m_r), (m_g + rO.m_g), (m_b + rO.m_b) );
  }
  
  /** Operator -. Substracts the color-component from each other.
   */
  inline Color operator - (const Color& rO) const
  {
    return Color( m_r - rO.m_r, m_g - rO.m_g, m_b - rO.m_b );
  }

  /** Operator +=. Adds another color to this color.
   * @param   aColor            Another color
   * @return  Reference to this Color
   */
  inline Color& operator += ( const Color& aColor )
  {
    m_r += aColor.m_r;
    m_g += aColor.m_g;
    m_b += aColor.m_b;
    return *this;
  }
  

  ///
  /// Methods
  /// =======
  
  /** Sets the rgb-values.
   * @param     red               New red component.
   * @param     green             New green component.
   * @param     blue              New blue component.
   */
  inline void setRGB(CVal red, CVal green, CVal blue)
  {
    m_r = red;
    m_g = green;
    m_b = blue;
  }
  
  /** Takes the color-components from given integer.
   * @param     allCol            Should have colorvalue in a correct
   *                              format so that we can take the color
   *                              components with Allegro's getr(...), etc.
   *                              functions.
   */
  inline void setRGB(int allCol)
  {
    m_r = getr( allCol );
    m_g = getg( allCol );
    m_b = getb( allCol );
  }
  
  
  /** Sets the red component.
   * @param     red               New red component.
   */
  inline void r( CVal red )
  {
    m_r = red;
  }
  
  /** Sets the green component.
   * @param     green             New green component.
   */
  inline void g( CVal green )
  {
    m_g = green;
  }
  
  /** Sets the blue component.
   * @param     blue              New blue component.
   */
  inline void b( CVal blue )
  {
    m_b = blue;
  }
  
  /** Changes the color component values by given amount of percents.
   * @param     percents          If <100 the color comes darker.
   *                              If >100 the color comes brighter.
   */
  inline void scale( int percents )
  {
    m_r = ( m_r * percents ) / 100;
    m_g = ( m_g * percents ) / 100;
    m_b = ( m_b * percents ) / 100;
  }
  
  /** Multiplies the color components with given value. Almost the same
   * as <code>void scale( int percents )</code>. Likely to become obsoled
   * later.
   * @param     f                 The multiplier.
   */
  inline void multiply(float f)
  {
    m_r = static_cast<int>( static_cast<float>(m_r) * f ) & 255;
    m_g = static_cast<int>( static_cast<float>(m_g) * f ) & 255;
    m_b = static_cast<int>( static_cast<float>(m_b) * f ) & 255;
  }
  
  
  ///
  /// Getter methods
  /// ==============
  

  /** Returns the colorvalue in one integer so that you can
   * use it with Allegro's drawing functions.
   * @return    Color
   */
  inline int allC() const
  {
    return makecol( static_cast<int>( m_r ), static_cast<int>( m_g ),
                    static_cast<int>( m_b ) );
  }
  
  /** Returns the red component.
   * @return    The red component.
   */
  inline CVal r() const { return m_r; }
  
  /** Returns the green component.
   * @return    The green component.
   */
  inline CVal g() const { return m_g; }
  
  /** Returns the blue component.
   * @return    The blue component.
   */
  inline CVal b() const { return m_b; }
  
};

};  // end of namespace

#endif
